#ifndef MENU_H
#define MENU_H

#include <stdint.h>
#include "SSD1306.h"

// Forward declaration of Menu struct is needed for the MenuItem union.
struct Menu;

// An Action is a function pointer that takes no arguments and returns nothing.
typedef void (*Action)(void);

// Each menu item can be either a link to a submenu or an action to execute.
typedef enum {
    MENU_ITEM_TYPE_SUBMENU,
    MENU_ITEM_TYPE_ACTION
} MenuItemType;

// Structure representing a single item in a menu.
typedef struct {
    const char* name;       // The text to display for the item.
    MenuItemType type;      // The type of the item (SUBMENU or ACTION).
    union {
        struct Menu* submenu; // Pointer to the submenu to open.
        Action action;        // Function pointer to the action to execute.
    } target;
} MenuItem;

// Structure representing a menu.
typedef struct Menu {
    const char* title;          // The title of the menu.
    const MenuItem* items;      // Pointer to an array of menu items.
    uint8_t num_items;          // The number of items in the menu.
    uint8_t selected_item;      // The index of the currently selected item.
    uint8_t scroll_offset;      // The scroll offset for display.
    struct Menu* parent;        // Pointer to the parent menu (for going back).
} Menu;

// --- Public API ---

// Initializes the menu system.
void menu_init(void);

// Renders the currently active menu on the screen.
void display_current_menu(void);

// Navigates to the next item in the current menu.
void menu_next(void);

// Navigates to the previous item in the current menu.
void menu_prev(void);

// Executes the action for the selected item or enters a submenu.
void menu_select(void);

// Navigates back to the parent menu.
void menu_back(void);

#endif // MENU_H
