#include "menu.h"
#include "actions.h"
#include "SSD1306.h"
#include "buzzer.h"
#include "led.h"      // Include for LED control
#include <stddef.h>
#include <util/delay.h>

// --- Action Prototypes ---
void action_media_play_pause(void) { send_media_command(CMD_PLAY_PAUSE); }
void action_media_next(void) { send_media_command(CMD_NEXT_TRACK); }
void action_media_prev(void) { send_media_command(CMD_PREV_TRACK); }
void action_media_vol_up(void) { send_media_command(CMD_VOLUME_UP); }
void action_media_vol_down(void) { send_media_command(CMD_VOLUME_DOWN); }
void action_media_mute(void) { send_media_command(CMD_MUTE); }

void action_general_copy(void) { send_general_command(CMD_COPY); }
void action_general_paste(void) { send_general_command(CMD_PASTE); }
void action_general_select_all(void) { send_general_command(CMD_SELECT_ALL); }
void action_general_search(void) { send_general_command(CMD_SEARCH); }

// --- Menu Definitions ---

// Forward declare menus
extern Menu main_menu;
extern Menu media_menu;
extern Menu settings_menu;
extern Menu code_menu;

// --- General Menu ---
const MenuItem general_menu_items[] = {
    {"Copy", MENU_ITEM_TYPE_ACTION, .target.action = &action_general_copy},
    {"Paste", MENU_ITEM_TYPE_ACTION, .target.action = &action_general_paste},
    {"Select all", MENU_ITEM_TYPE_ACTION, .target.action = &action_general_select_all},
    {"Search", MENU_ITEM_TYPE_ACTION, .target.action = &action_general_search},
};
Menu general_menu = {
    .title = "General",
    .items = general_menu_items,
    .num_items = sizeof(general_menu_items) / sizeof(MenuItem),
    .selected_item = 0,
    .scroll_offset = 0,
    .parent = &main_menu
};

// --- Media Menu ---
const MenuItem media_menu_items[] = {
    {"Play/Pause", MENU_ITEM_TYPE_ACTION, .target.action = &action_media_play_pause},
    {"Next Track", MENU_ITEM_TYPE_ACTION, .target.action = &action_media_next},
    {"Prev Track", MENU_ITEM_TYPE_ACTION, .target.action = &action_media_prev},
    {"Volume +",   MENU_ITEM_TYPE_ACTION, .target.action = &action_media_vol_up},
    {"Volume -",   MENU_ITEM_TYPE_ACTION, .target.action = &action_media_vol_down},
    {"Mute",       MENU_ITEM_TYPE_ACTION, .target.action = &action_media_mute},
};
Menu media_menu = {
    .title = "Media",
    .items = media_menu_items,
    .num_items = sizeof(media_menu_items) / sizeof(MenuItem),
    .selected_item = 0,
    .scroll_offset = 0,
    .parent = &main_menu
};

// --- Main Menu ---
const MenuItem main_menu_items[] = {
    {"Media", MENU_ITEM_TYPE_SUBMENU, .target.submenu = &media_menu},
    {"General", MENU_ITEM_TYPE_SUBMENU, .target.submenu = &general_menu},
};
Menu main_menu = {
    .title = "Main Menu",
    .items = main_menu_items,
    .num_items = sizeof(main_menu_items) / sizeof(MenuItem),
    .selected_item = 0,
    .scroll_offset = 0,
    .parent = NULL
};

// Pointer to the currently active menu.
static Menu* current_menu;

// --- Private Functions ---
static void display_action_feedback(const char* message) {
    GLCD_Clear();
    GLCD_GotoXY(10, 28);
    GLCD_PrintString((char*)message);
    GLCD_Render();
    
    all_led_on();
    buzzer_on();
    _delay_ms(200);
    buzzer_off();
    all_led_off();
}

// --- Public API Implementation ---
void menu_init(void) {
    current_menu = &main_menu;
}

void display_current_menu(void) {
    GLCD_Clear();
    GLCD_GotoXY(2, 2);
    GLCD_PrintString((char*)current_menu->title);

    const uint8_t FONT_HEIGHT = 8;
    const uint8_t ITEM_PADDING = 3;
    const uint8_t rect_height = FONT_HEIGHT + ITEM_PADDING * 2;
    const uint8_t max_visible_items = 3;

    for (uint8_t i = 0; i < max_visible_items; i++) {
        uint8_t item_index = current_menu->scroll_offset + i;
        if (item_index >= current_menu->num_items) break;

        uint8_t y_position = 15 + i * rect_height;
        if (item_index == current_menu->selected_item) {
            GLCD_DrawRectangle(1, y_position, 126, y_position + rect_height - 1, GLCD_Black);
            GLCD_GotoXY(5, y_position + ITEM_PADDING);
            GLCD_PrintString((char*)current_menu->items[item_index].name);
        } else {
            GLCD_GotoXY(5, y_position + ITEM_PADDING);
            GLCD_PrintString((char*)current_menu->items[item_index].name);
        }
    }
    GLCD_Render();
}

void menu_next(void) {
    if (current_menu->selected_item < current_menu->num_items - 1) {
        current_menu->selected_item++;
        if (current_menu->selected_item >= current_menu->scroll_offset + 3) {
            current_menu->scroll_offset++;
        }
    }
}

void menu_prev(void) {
    if (current_menu->selected_item > 0) {
        current_menu->selected_item--;
        if (current_menu->selected_item < current_menu->scroll_offset) {
            current_menu->scroll_offset = current_menu->selected_item;
        }
    }
}

void menu_select(void) {
    const MenuItem* selected = &current_menu->items[current_menu->selected_item];
    if (selected->type == MENU_ITEM_TYPE_SUBMENU && selected->target.submenu != NULL) {
        current_menu = selected->target.submenu;
    } else if (selected->type == MENU_ITEM_TYPE_ACTION && selected->target.action != NULL) {
        selected->target.action();
        display_action_feedback(selected->name);
    }
}

void menu_back(void) {
    if (current_menu->parent != NULL) {
        current_menu = current_menu->parent;
    }
}
