/*
             LUFA Library
     Copyright (C) Dean Camera, 2021.

  dean [at] fourwalledcubicle [dot] com
           www.lufa-lib.org
*/

#define  INCLUDE_FROM_MINIMAL_C
#include "Binioukey.h"
#include "projectfiles/StorageHandler.h"
#include "projectfiles/Action.h"
#include "projectfiles/Display.h"
#include "projectfiles/Button.h"
#include "projectfiles/Generation.h"
#include "Constants.h"
#include "projectfiles/Usb.h"
#include "../../lib/LUFA/Drivers/USB/USB.h"

// Request de génération de clé depuis l'endpoint de contrôle
volatile bool generate_key_requested = false;

int main(void)
{
    // Setup de LUFA
    SetupHardware();
    GlobalInterruptEnable();

    // Setup de nous
    setup_all();
    setup_screen();
    setup_buttons();
    display_title();
    display_buttons_usage();

    // +1 pour le caractère nul de fin de chaine '\0'
    char key_buffer[KEY_LENGTH + 1]; 
    char screen_buffer[21];
    
    uint8_t last_stored_key_index = NO_KEY_SELECTED;

    // Boucle principale
    for (;;)
    {

        USB_USBTask();

    	// On a reçu une demande de génération de clé : on le fait
    	if (generate_key_requested) {

    		const int status = generate_and_save(key_buffer);

    		generate_key_requested = false;
    		continue;
    	}

        // --- BOUTON GAUCHE : RESET ---
        if (is_left_button_pressed()) {
			clear_body();
            reset_storage();
            last_stored_key_index = NO_KEY_SELECTED;
            
            display_string("The storage has been", 6);
			display_string("successfully cleaned!", 7);
            _delay_ms(2000);
            display_buttons_usage();
        }

        // --- BOUTON HAUT : GÉNÉRER ET SAUVEGARDER ---
        if (is_up_button_pressed()) {
			clear_body();
            const uint8_t save_index = generate_and_save(key_buffer);

            if (save_index != 0xff) {
                // Succès
                snprintf(screen_buffer, sizeof(screen_buffer), "Saved at ID: %u", save_index);
                display_string(screen_buffer, 5);
                display_string("Key:", 6);
                
                key_buffer[KEY_LENGTH] = '\0';	// Pour éviter les débordements à l'affichage
				display_string(key_buffer, 7);
            } else {
                // Erreur : Mémoire pleine
                display_string("Error: Memory full!  ", 6);
                display_string("Cannot save.         ", 7);
            }

            _delay_ms(3000);
            display_buttons_usage();
        }

        // --- BOUTON CENTRAL : ENVOYER LA CLE ACTIVE ---
        if (is_center_button_pressed()) {
			clear_body();
			if (last_stored_key_index == NO_KEY_SELECTED) {
				display_string("No selected key.     ", 5);
				display_string("Select one with the  ", 6);
				display_string("right button.        ", 7);
			} else {
				snprintf(screen_buffer, sizeof(screen_buffer), "Key no. %u", last_stored_key_index);
				display_string(screen_buffer, 3);

				read_key(last_stored_key_index, key_buffer);

				handle_in(key_buffer);

				key_buffer[KEY_LENGTH] = '\0';	// Pour éviter les débordements à l'affichage
				display_string(key_buffer, 4);

				display_string("has been sent to     ", 6);
				display_string("SarrasinDefender!    ", 7);
			}
			
			_delay_ms(3000);
            display_buttons_usage();
        }

		// --- BOUTON DROIT : SELECTIONNER LA PROCHAINE CLE ---
        if (is_right_button_pressed()) {
			clear_body();

			uint8_t bitmap = get_table_base_addr();

			if (bitmap == 0x00) {
				// Aucune clé sauvegardée
				display_string("No key found.        ", 5);
				display_string("Create one by using  ", 6);
				display_string("the up button.       ", 7);
			} else {

				uint8_t result = last_stored_key_index;

				// Remise à zéro du pointeur si on a atteint la fin du tableau
				// ou s'il vaut NO_KEY_SELECTED (au démarrage de la BiniouKey)
				if (result == NO_KEY_SELECTED || result == MAX_KEYS - 1) {
					result = -1;
				} 

				++result;
			
				while (result < MAX_KEYS) {
					if (bitmap & (1 << result)) {
						break;
					}
					++result;
				}

				if (result >= MAX_KEYS) {
					// Fin de liste atteint
					last_stored_key_index = NO_KEY_SELECTED;
					clear_body();
					display_string("End of list.         ", 4);
					display_string("Wait for the menu and", 5);
					display_string("press again the right", 6);
					display_string("button to reloop.    ", 7);
				} else {
					last_stored_key_index = result;		
					snprintf(screen_buffer, sizeof(screen_buffer), "Key no. %u selected.", last_stored_key_index);
					display_string(screen_buffer, 5);

					read_key(last_stored_key_index, key_buffer);
					key_buffer[KEY_LENGTH] = '\0';	// Pour éviter les débordements à l'affichage

					display_string("Value: ", 6);
					display_string(key_buffer, 7);
				}

			}

			_delay_ms(3000);
			display_buttons_usage();
        }

		// --- BOUTON BAS : SUPPRIMER LA CLE SELECTIONNEE ---
        if (is_down_button_pressed()) {
			clear_body();

			if (last_stored_key_index == NO_KEY_SELECTED) {
				display_string("No selected key.     ", 5);
				display_string("Select one with the  ", 6);
				display_string("right button.        ", 7);
			} else {
				snprintf(screen_buffer, sizeof(screen_buffer), "Key no. %u", last_stored_key_index);
				display_string(screen_buffer, 3);

				read_key(last_stored_key_index, key_buffer);
				key_buffer[KEY_LENGTH] = '\0';	// Pour éviter les débordements à l'affichage
				display_string(key_buffer, 4);

				display_string("has been             ", 6);
				display_string("deleted successfully!", 7);

				delete_key(last_stored_key_index);
			}
			
			_delay_ms(3000);
            display_buttons_usage();
		}
    }
}

/** Configures the board hardware and chip peripherals for the demo's functionality. */
void SetupHardware(void)
{
#if (ARCH == ARCH_AVR8)
    MCUSR &= ~(1 << WDRF);
    wdt_disable();
    clock_prescale_set(clock_div_1);
#elif (ARCH == ARCH_XMEGA)
    XMEGACLK_StartPLL(CLOCK_SRC_INT_RC2MHZ, 2000000, F_CPU);
    XMEGACLK_SetCPUClockSource(CLOCK_SRC_PLL);
    XMEGACLK_StartInternalOscillator(CLOCK_SRC_INT_RC32MHZ);
    XMEGACLK_StartDFLL(CLOCK_SRC_INT_RC32MHZ, DFLL_REF_INT_USBSOF, F_USB);
    PMIC.CTRL = PMIC_LOLVLEN_bm | PMIC_MEDLVLEN_bm | PMIC_HILVLEN_bm;
#endif
    USB_Init();
}

// Les event handlers USB restent inchangés...
void EVENT_USB_Device_Connect(void) {}
void EVENT_USB_Device_Disconnect(void) {}
void EVENT_USB_Device_ConfigurationChanged(void)
{
    Endpoint_ConfigureEndpoint(VENDOR_IN_EPADDR, EP_TYPE_INTERRUPT, VENDOR_IN_EPSIZE, 1);
    Endpoint_ConfigureEndpoint(VENDOR_OUT_EPADDR, EP_TYPE_INTERRUPT, VENDOR_OUT_EPSIZE, 1);
}

/** Event handler for the USB_ControlRequest event. This is used to catch and process control requests sent to
 *  the device from the USB host before passing along unhandled control requests to the library for processing
 *  internally.
 */
void EVENT_USB_Device_ControlRequest(void)
{
	// 1. Request type est de notre vendor, et le requête vient de l'host vers nous
	if (((USB_ControlRequest.bmRequestType & (CONTROL_REQTYPE_TYPE | CONTROL_REQTYPE_DIRECTION)) ==
		(REQTYPE_VENDOR | REQDIR_HOSTTODEVICE)))
	{
		// 2. Vérification du type de commande (définie par la même constante)
		if (USB_ControlRequest.bRequest == CONTROL_ENDPOINT_CMD_GENERATE)
		{
			// 3. Endpoint 0
			Endpoint_SelectEndpoint(ENDPOINT_CONTROLEP);

			// Si on avait des données à lire on le ferait ici
			// là on en a pas donc pas besoin de lire quoi que ce soit
			// pratique parce que c'est le genre de truc chiant

			// 4. Acknowledge du message
			Endpoint_ClearSETUP();
			Endpoint_ClearStatusStage();

			// On met le flag à true (la génération est faite dans la boucle principale)
			generate_key_requested = true;
		}
	}
}