//
// Created by léo on 25/10/2025.
//

#include "Minuteur.h"

#include <avr/io.h>       // Nécessaire pour accéder aux registres AVR (TCCR1A, PORTB, etc.)
#include <avr/interrupt.h> // Nécessaire pour utiliser les interruptions (ISR, sei)

// ============================================================================
// DÉFINITIONS ET CONSTANTES
// ============================================================================

// CTC = Clear Timer on Compare match
// Ce mode réinitialise automatiquement le compteur à 0 quand il atteint la valeur de comparaison
#define CTC1            WGM12

// Période par défaut pour l'exemple LED (en millisecondes)
#define PERIODE         1000

// ============================================================================
// POINTEUR DE FONCTION CALLBACK
// ============================================================================

// Ce pointeur de fonction permet de définir ce qui doit être exécuté à chaque interruption
// Pour utiliser ce timer dans un autre contexte, il suffit de créer une fonction
// avec la signature void fonction(void) et de la passer à init_minuteur_avec_callback()
void (*timer_callback)(void) = 0;

// ============================================================================
// FONCTION D'INITIALISATION DU TIMER 1
// ============================================================================

/**
 * Configure le Timer1 de l'AVR en mode CTC (Clear Timer on Compare)
 *
 * @param diviseur : Le facteur de division de l'horloge (prescaler)
 *                   Valeurs possibles : 8, 64, 256, 1024
 *                   Plus le diviseur est grand, plus le timer est lent
 *                   mais peut compter jusqu'à des périodes plus longues
 *
 * @param periode : La période d'interruption en millisecondes
 *                  À chaque 'periode' ms, l'interruption ISR(TIMER1_COMPA_vect) sera déclenchée
 *
 * FONCTIONNEMENT DU TIMER:
 * - Le Timer1 est un compteur 16 bits (valeur de 0 à 65535)
 * - Il compte les cycles d'horloge divisés par le prescaler
 * - Quand il atteint la valeur OCR1A, il déclenche une interruption et se remet à 0
 */
void init_minuteur(int diviseur, long periode) {
    // TCCR1A = Timer/Counter1 Control Register A
    // On le met à 0 car en mode CTC, on n'utilise pas les fonctionnalités de ce registre
    // (comme la génération de PWM sur les pins de sortie)
    TCCR1A = 0;

    // TCCR1B = Timer/Counter1 Control Register B
    // Configuration du mode CTC : le timer se réinitialise quand TCNT1 == OCR1A
    // WGM12 (bit 3) doit être à 1 pour activer le mode CTC
    TCCR1B = (1 << CTC1);

    // Configuration du PRESCALER (diviseur d'horloge)
    // Le prescaler ralentit l'horloge CPU avant qu'elle n'atteigne le timer
    // Les bits CS12, CS11, CS10 dans TCCR1B déterminent le facteur de division
    //
    // Table de vérité des bits CS (Clock Select) :
    // CS12 CS11 CS10 | Prescaler
    //  0    0    0   | Pas d'horloge (timer arrêté)
    //  0    0    1   | /1 (pas de division)
    //  0    1    0   | /8
    //  0    1    1   | /64
    //  1    0    0   | /256
    //  1    0    1   | /1024
    switch (diviseur) {
        case 8:
            // CS11 = 1, les autres à 0 → prescaler = 8
            TCCR1B |= (1 << CS11);
            break;
        case 64:
            // CS11 = 1, CS10 = 1 → prescaler = 64
            TCCR1B |= (1 << CS11 | 1 << CS10);
            break;
        case 256:
            // CS12 = 1, les autres à 0 → prescaler = 256
            TCCR1B |= (1 << CS12);
            break;
        case 1024:
            // CS12 = 1, CS10 = 1 → prescaler = 1024
            TCCR1B |= (1 << CS12 | 1 << CS10);
            break;
    }

    // CALCUL DE LA VALEUR DE COMPARAISON (OCR1A)
    //
    // Explications détaillées du calcul :
    //
    // 1. L'horloge CPU tourne à F_CPU Hz (par exemple 16 000 000 Hz = 16 MHz)
    //    → Un cycle d'horloge dure 1/F_CPU secondes
    //
    // 2. Avec le prescaler, un "tick" du timer dure diviseur/F_CPU secondes
    //    Exemple avec F_CPU=16MHz et diviseur=256 : 256/16000000 = 0.000016 secondes = 16 µs
    //
    // 3. On veut une période en millisecondes (periode)
    //    → période en secondes = periode / 1000
    //
    // 4. Nombre de ticks nécessaires = (periode_en_secondes) / (duree_d_un_tick)
    //                                = (periode/1000) / (diviseur/F_CPU)
    //                                = (periode/1000) * (F_CPU/diviseur)
    //                                = (periode * F_CPU) / (1000 * diviseur)
    //
    // Exemple concret : F_CPU=16MHz, diviseur=256, periode=1000ms
    // OCR1A = (1000 * 16000000) / (1000 * 256) = 62500
    // Le timer comptera jusqu'à 62500 avant de déclencher l'interruption
    OCR1A = F_CPU / 1000 * periode / diviseur;

    // TCNT1 = Timer/Counter1 Register
    // C'est le registre qui contient la valeur actuelle du compteur
    // On l'initialise à 0 pour démarrer proprement
    TCNT1 = 0;

    // TIMSK1 = Timer/Counter1 Interrupt Mask Register
    // Ce registre active/désactive les différentes interruptions du Timer1
    // OCIE1A (Output Compare Interrupt Enable 1 A) = bit qui active l'interruption
    // quand TCNT1 atteint la valeur OCR1A
    TIMSK1 = (1 << OCIE1A);
}

// ============================================================================
// VERSION AVEC CALLBACK (POUR UTILISATION GÉNÉRIQUE)
// ============================================================================

/**
 * Version améliorée de init_minuteur qui permet de spécifier une fonction callback
 * Cette fonction sera appelée à chaque interruption du timer
 *
 * @param diviseur : même chose que init_minuteur
 * @param periode : même chose que init_minuteur
 * @param callback : pointeur vers une fonction void callback(void) qui sera
 *                   exécutée à chaque interruption
 *
 * EXEMPLE D'UTILISATION :
 *
 * void ma_fonction_periodique(void) {
 *     // Code à exécuter toutes les X millisecondes
 *     PORTD ^= (1 << PD0); // Toggle une LED sur PD0 par exemple
 * }
 *
 * int main(void) {
 *     init_minuteur_avec_callback(256, 500, ma_fonction_periodique); // Appel toutes les 500ms
 *     sei(); // Active les interruptions globalement
 *     while(1);
 * }
 */
void init_minuteur_avec_callback(int diviseur, long periode, void (*callback)(void)) {
    // Sauvegarde la fonction callback pour l'appeler dans l'ISR
    timer_callback = callback;

    // Initialise le timer normalement
    init_minuteur(diviseur, periode);
}

// ============================================================================
// ROUTINE D'INTERRUPTION (ISR = Interrupt Service Routine)
// ============================================================================

/**
 * ISR = Interrupt Service Routine
 * Cette fonction est appelée AUTOMATIQUEMENT par le hardware quand le Timer1
 * atteint la valeur OCR1A (grâce au mode CTC activé précédemment)
 *
 * TIMER1_COMPA_vect = vecteur d'interruption pour la comparaison A du Timer1
 *
 * IMPORTANT :
 * - Cette fonction doit être TRÈS RAPIDE (pas de delay, pas de calculs longs)
 * - Elle est exécutée même si le programme principal est occupé ailleurs
 * - Les interruptions sont automatiquement désactivées pendant son exécution
 */
ISR(TIMER1_COMPA_vect) {
    // Si un callback a été défini, on l'appelle
    if (timer_callback != 0) {
        timer_callback();
    }
    // Sinon, on exécute le comportement par défaut (chenillard LED)
    else {
        led_chenillard();
    }
}

// ============================================================================
// FONCTION EXEMPLE : CHENILLARD LED
// ============================================================================

/**
 * Fonction exemple qui fait défiler une LED sur 4 pins (chenillard)
 * Cette fonction peut être remplacée par n'importe quelle autre fonction
 * selon vos besoins
 *
 * FONCTIONNEMENT DU CHENILLARD :
 * - Lit l'état actuel des 4 LEDs sur les 4 bits de poids faible du PORTB
 * - Décale la LED allumée d'une position vers la droite
 * - Revient au début quand on atteint la fin
 */
void led_chenillard(void) {
    // PINB = Pin Input Register B
    // Permet de LIRE l'état actuel des pins du PORTB
    // & 0x0f = masque pour garder seulement les 4 bits de poids faible (bits 0-3)
    // 0x0f en binaire = 0b00001111
    int led = (PINB & 0x0f);

    // Décalage à droite d'une position (>>= 1)
    // Si led = 0b00001000, après décalage led = 0b00000100
    led >>= 1;

    // Si on arrive à 0 (toutes les LEDs éteintes), on revient au début
    // On rallume la LED la plus à gauche (bit 3)
    if (led == 0) led = 0x08; // 0x08 = 0b00001000

    // PORTB = Port B Data Register
    // Permet d'ÉCRIRE sur les pins configurées en sortie
    // On efface d'abord les 4 bits de poids faible (éteint les LEDs)
    PORTB &= 0xf0; // 0xf0 = 0b11110000 (garde les 4 bits de poids fort)

    // Puis on allume la nouvelle LED
    PORTB |= led; // | = OR logique, active les bits correspondants
}

// ============================================================================
// FONCTION PRINCIPALE (EXEMPLE D'UTILISATION)
// ============================================================================

/**
 * Fonction main - Point d'entrée du programme
 * Configure les LEDs et démarre le timer périodique
 */
int main(void) {
    // DDRB = Data Direction Register B
    // Détermine si chaque pin du PORTB est en entrée (0) ou sortie (1)
    // |= 0x0f configure les 4 bits de poids faible en SORTIE
    // 0x0f = 0b00001111 → pins PB0, PB1, PB2, PB3 en sortie pour les LEDs
    DDRB |= 0x0f;

    // PORTB = Port B Data Register
    // Quand une pin est en sortie, écrire dans PORTB met la pin à HIGH (1) ou LOW (0)
    // &= ~0x0f met les 4 bits de poids faible à 0 → éteint toutes les LEDs au démarrage
    // ~ = NOT binaire : ~0x0f = 0b11110000
    PORTB &= ~0x0f;

    // EXEMPLE 1 : Utilisation avec le callback par défaut (chenillard)
    init_minuteur(256, PERIODE);

    // EXEMPLE 2 : Utilisation avec une fonction personnalisée
    // Décommenter les lignes suivantes et commenter la ligne au-dessus
    // pour utiliser votre propre fonction
    /*
    void ma_fonction(void) {
        // Votre code ici
        PORTB ^= (1 << PB0); // Fait clignoter la LED sur PB0
    }
    init_minuteur_avec_callback(256, PERIODE, ma_fonction);
    */

    // sei() = Set Enable Interrupt
    // Active les interruptions GLOBALEMENT sur l'AVR
    // Sans cet appel, l'ISR ne sera jamais exécutée même si le timer est configuré
    // C'est comme un interrupteur général pour toutes les interruptions
    sei();

    // Boucle infinie
    // Le programme reste ici indéfiniment
    // Pendant ce temps, l'ISR s'exécute automatiquement toutes les PERIODE millisecondes
    // grâce au système d'interruptions
    while (1);
}
