#include "StorageHandler.h"
#include <string.h>

static uint8_t* key_addr(uint8_t index) {
    // Adresse Base + (Numéro Clé * Taille Clé)
    return EEPROM_KEYS_BASE_ADDR + (index * KEY_LENGTH);
}

uint8_t get_table_base_addr(void) {
    return eeprom_read_byte(EEPROM_TABLE_BASE_ADDR);
}

uint8_t get_next_free_index(void) {
    // Lire l'état actuel des 8 emplacements (le bitmap)
    uint8_t bitmap = get_table_base_addr();

    // Parcourir les 8 bits pour trouver le premier à 0
    for (uint8_t i = 0; i < MAX_KEYS; i++) {
        // Vérifier si le bit i est à 0. et le retourner
        // (1 << i) crée un masque (ex: 00000100 pour i=2)
        if ((bitmap & (1 << i)) == 0) {
            return i;
        }
    }

    return 0xFF; // Code d'erreur : Plus de place
}

uint8_t store_key(const char *buffer, uint8_t *idx) {
    uint8_t free_idx = get_next_free_index();

    if (free_idx == 0xFF) {
        return 1; // Erreur : EEPROM Pleine
    }

    // Écrire la clé à l'adresse calculée
    eeprom_update_block(buffer, key_addr(free_idx), KEY_LENGTH);

    // Mettre à jour le bitmap pour marquer 'free_idx' comme occupé
    uint8_t bitmap = get_table_base_addr();
    
    // Opérateur OU Bit à Bit (|) pour mettre le bit à 1
    bitmap |= (1 << free_idx); 
    
    // Mise à jour du bitmap dans EEPROM
    eeprom_update_byte(EEPROM_TABLE_BASE_ADDR, bitmap);

    // Retourner l'adresse de la clé sauvée
    if (idx) {
        *idx = free_idx;
    }
    return 0; // Succès
}

uint8_t read_key(uint8_t index, char *buffer) {
    if (index >= MAX_KEYS) return 0;

    // Récupérer la table des emplacements de stockage des clés
    uint8_t bitmap = get_table_base_addr();

    // Vérifier si le bit est à 1
    if (bitmap & (1 << index)) {
        // Le bit est à 1, on lit les données
        eeprom_read_block(buffer, key_addr(index), KEY_LENGTH);
        return 1; // Clé trouvée
    } else {
        // Le bit est à 0, l'emplacement est vide
        memset(buffer, 0, KEY_LENGTH);  // On bourre le buffer de 0 pour montrer que c'est une erreur
        return 0; // Rien à lire ici
    }
}

void delete_key(uint8_t index) {
    if (index >= MAX_KEYS) return;

    uint8_t bitmap = get_table_base_addr();

    // Opérateur ET avec l'inverse du masque (~(1<<i)) pour mettre le bit à 0
    // Ex: Masque 00000100 -> Inverse 11111011 -> AND force le 3ème bit à 0
    bitmap &= ~(1 << index);

    eeprom_update_byte(EEPROM_TABLE_BASE_ADDR, bitmap);
}

void reset_storage() {
    // Remise du bitmap à 0 
    // Les anciennes données des clés resteront en mémoire mais seront considérées comme "libres" et écrasables.
    eeprom_update_byte(EEPROM_TABLE_BASE_ADDR, 0x0);
}

