//
// Created by antame on 08/01/2026.
// Stocker et lire dans la EEPROM de la Binioukey :)
//

#ifndef BINIOUKEY_STORAGE_HANDLER_H
#define BINIOUKEY_STORAGE_HANDLER_H

#include "Constants.h"
#include <avr/io.h>
#include <avr/eeprom.h>

/**
 * Référenciel du stockage de clé, sauvegardé au début de la mémoire EEPROM.
 * Pour 8 clés, on a donc un octet (0000 0000) où chaque bit représente la disponibilité de l'adresse mémoire.
 * 
 * 0 = emplacement disponible ; 1 = emplacement occupé. 
 */
#define EEPROM_TABLE_BASE_ADDR ((uint8_t*) 0)

/**
 * Emplacement du premier caractère de la première clé, sauvegardé dans la mémoire EEPROM juste 
 * après EEPROM_TABLE_BASE_ADDR.
 */
#define EEPROM_KEYS_BASE_ADDR ((uint8_t*) 1)

/**
 * Lire la valeur EEPROM_TABLE_BASE_ADDR sauvegardée. 
 */
uint8_t get_table_base_addr(void);

/**
 * Retourne le prochain index libre (0-7).
 * Retourne 0xFF (255) si l'EEPROM est pleine.
 */
uint8_t get_next_free_index(void);

/**
 * Sauvegarder une clé au premier emplacement libre trouvé.
 * @param key16 La clé à sauvegarder.
 * @param idx Pointeur pour récupérer l'index où la clé a été stockée (optionnel).
 * @return 0 si succès, 1 si erreur (plus de place).
 */
uint8_t store_key(const char *key16, uint8_t *idx);

/**
 * Obtenir une clé sauvegardée.
 * @param index L'indice de la clé.
 * @param buffer16 Le buffer de destination.
 * @return 1 si une clé a été trouvée, 0 si l'emplacement est vide.
 */
uint8_t read_key(uint8_t index, char *buffer16);

/**
 * Supprimer une clé spécifique.
 * Pour éviter un traitement lourd, la clé indique que l'emplacement est libre.
 */
void delete_key(uint8_t index);

/**
 * Effacer la mémoire.
 * Pour éviter un traitement lourd, la clé remet juste la valeur de 
 * EEPROM_TABLE_BASE_ADDR à 0 (i.e. tous les emplacements sont libres).
 */
void reset_storage();

#endif //BINIOUKEY_STORAGE_HANDLER_H