/*
             LUFA Library
     Copyright (C) Dean Camera, 2021.

  dean [at] fourwalledcubicle [dot] com
           www.lufa-lib.org
*/

/*
  Copyright 2021  Dean Camera (dean [at] fourwalledcubicle [dot] com)

  Permission to use, copy, modify, distribute, and sell this
  software and its documentation for any purpose is hereby granted
  without fee, provided that the above copyright notice appear in
  all copies and that both that the copyright notice and this
  permission notice and warranty disclaimer appear in supporting
  documentation, and that the name of the author not be used in
  advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.

  The author disclaims all warranties with regard to this
  software, including all implied warranties of merchantability
  and fitness.  In no event shall the author be liable for any
  special, indirect or consequential damages or any damages
  whatsoever resulting from loss of use, data or profits, whether
  in an action of contract, negligence or other tortious action,
  arising out of or in connection with the use or performance of
  this software.
*/

/** \file
 *
 *  Main source file for the Minimal demo. This file contains the main tasks of the demo and
 *  is responsible for the initial application hardware configuration.
 */

#define  INCLUDE_FROM_MINIMAL_C
#include "Minimal.h"
#include "../../LUFA/LUFA/Drivers/USB/Core/USBController.h"

/* Project peripherals used by the demo (LCD, servo, sonar) */
#include <stdio.h>
#include <stdlib.h>
#include <util/delay.h>
#include "sonar/lcd/HD44780.h"
#include "sonar/servo.h"
#include "sonar/sonar.h"

/* Hardware pin/feature defines used by the original sonar demo */
#define LED1        1    // PB1
#define LED2        2    // PB2
#define LED3        7    // PB7
#define LED4        6    // PE7
#define BOUTON_G    3    // PB3
#define BOUTON_C    7    // PC7
#define DELAY_time  300  // temps entre mesures
#define DEBOUNCE_MS 20



/* Application state */
static uint8_t app_autoMode = 0;  /* Start in USB Control mode (Python controls servo) */
static uint8_t app_lastC = 1;
static uint8_t app_lastG = 1;
static int8_t app_direction = 1;
static char app_buffer[17];

/* Initialize application hardware and perform a small startup check */
void App_Init(void)
{
  SetupHardware();
  GlobalInterruptEnable();

  /* startup indicators */
  PORTB |= (1 << LED1);
  _delay_ms(150);
  PORTB &= ~(1 << LED1);

  HD44780_WriteCommand(LCD_HOME);
  HD44780_DisplayLine(0, "BOOT OK");
  _delay_ms(300);

  /* small servo move to validate PWM output */
  setServoAngle(10);
  _delay_ms(300);
  setServoAngle(5);
  _delay_ms(200);
}

/* Application main tasks — called from main loop */
void App_Tasks(void)
{
  /* USB housekeeping - doit être appelé régulièrement */
  USB_USBTask();

  /* --- LOCAL UI --- */
  uint8_t curC = (PINC & (1 << BOUTON_C)) ? 1 : 0;
  uint8_t curG = (PINB & (1 << BOUTON_G)) ? 1 : 0;

  /* Bouton G : incrément manuel */
  if (app_lastG == 1 && curG == 0) {
    _delay_ms(DEBOUNCE_MS);
    if (!(PINB & (1 << BOUTON_G))) {
      app_autoMode = 0; 
      uint32_t a = getAngle();
      a++;
      if (a > 19) a = 5;
      setServoAngle(a);
      _delay_ms(50);
      uint32_t dist = sonar_mesure();
      char buf[17];
      snprintf(buf, sizeof(buf), "A:%2lu D:%3lucm", (unsigned long)a, (unsigned long)dist);
      HD44780_WriteCommand(LCD_HOME);
      HD44780_DisplayLine(0, buf);
    }
  }

  /* Bouton C : toggle auto */
  if (app_lastC == 1 && curC == 0) {
    _delay_ms(DEBOUNCE_MS);
    if (!(PINC & (1 << BOUTON_C))) {
      app_autoMode = !app_autoMode;
      if (app_autoMode) {
        HD44780_WriteCommand(LCD_HOME);
        HD44780_DisplayLine(0, "Mode Auto ON    ");
      } else {
        HD44780_WriteCommand(LCD_HOME);
        HD44780_DisplayLine(0, "USB Control     ");
      }
      _delay_ms(500);
    }
  }

  /* === MODE USB CONTROLLED (Python envoie les angles) === */
  if (!app_autoMode) {
    // Vérifier si on a reçu une commande d'angle via USB OUT endpoint
    Endpoint_SelectEndpoint(ENDPOINT_DIR_OUT | 2);
    if (Endpoint_IsOUTReceived()) {
      char rxbuf[64];
      int idx = 0;
      
      // Lire les données reçues
      while (Endpoint_IsReadWriteAllowed() && idx < 63) {
        rxbuf[idx++] = Endpoint_Read_8();
      }
      rxbuf[idx] = '\0';
      Endpoint_ClearOUT();
      
      // Parser la commande "S:XX" pour l'angle
      if (rxbuf[0] == 'S' && rxbuf[1] == ':') {
        int newAngle = 0;
        for (int i = 2; rxbuf[i] && rxbuf[i] != '\n' && rxbuf[i] != '\r'; i++) {
          if (rxbuf[i] >= '0' && rxbuf[i] <= '9') {
            newAngle = newAngle * 10 + (rxbuf[i] - '0');
          }
        }
        
        // Appliquer l'angle au servo
        if (newAngle >= 5 && newAngle <= 19) {
          setServoAngle((uint8_t)newAngle);
          _delay_ms(50);  // Stabilisation servo
          
          // Mesurer distance
          uint16_t dist = sonar_mesure();
          
          // Afficher sur LCD
          char buf[17];
          if (dist > 2 && dist < 400)
            snprintf(buf, sizeof(buf), "A:%2d D:%3ucm    ", newAngle, dist);
          else
            snprintf(buf, sizeof(buf), "A:%2d D:---      ", newAngle);
          HD44780_WriteCommand(LCD_HOME);
          HD44780_DisplayLine(0, buf);
          
          // Envoyer la réponse via USB IN
          Endpoint_SelectEndpoint(ENDPOINT_DIR_IN | 1);
          if (Endpoint_IsINReady()) {
            char txbuf[64];
            int len = snprintf(txbuf, sizeof(txbuf), "A:%d D:%u\n", newAngle, dist);
            for (int i = 0; i < len; ++i)
              Endpoint_Write_8((uint8_t)txbuf[i]);
            Endpoint_ClearIN();
          }
        }
      }
    }
  }

  /* === AUTO MODE SWEEP (balayage autonome sur la carte) === */
  if (app_autoMode) {
    static uint8_t angle = 5;
    
    setServoAngle(angle);
    uint16_t dist = sonar_mesure();
    
    char buf[17];
    if (dist > 2 && dist < 400)
      snprintf(buf, sizeof(buf), "A:%2u D:%3ucm    ", angle, dist);
    else
      snprintf(buf, sizeof(buf), "A:%2u D:---      ", angle);
    HD44780_WriteCommand(LCD_HOME);
    HD44780_DisplayLine(0, buf);
    
    Endpoint_SelectEndpoint(ENDPOINT_DIR_IN | 1);
    if (Endpoint_IsINReady()) {
        char txbuf[64];
        int len = snprintf(txbuf, sizeof(txbuf), "A:%u D:%u\n", angle, dist);
        for (int i = 0; i < len; ++i)
            Endpoint_Write_8((uint8_t)txbuf[i]);
        Endpoint_ClearIN();
    }
    
    angle += app_direction;
    if (angle >= 19) {
        angle = 19;
        app_direction = -1;
    } else if (angle <= 5) {
        angle = 5;
        app_direction = 1;
    }
    
    _delay_ms(DELAY_time);
  }

  app_lastC = curC;
  app_lastG = curG;
}

int main(void)
{
  App_Init();
  for (;;)
  {
    App_Tasks();
  }
}

/** Configures the board hardware and chip peripherals for the demo's functionality. */
void SetupHardware(void)
{
#if (ARCH == ARCH_AVR8)
	/* Disable watchdog if enabled by bootloader/fuses - MUST BE FIRST */
	MCUSR &= ~(1 << WDRF);
	wdt_disable();

	/* Disable clock division - MUST BE BEFORE ANY TIMER SETUP */
	clock_prescale_set(clock_div_1);
#elif (ARCH == ARCH_XMEGA)
	XMEGACLK_StartPLL(CLOCK_SRC_INT_RC2MHZ, 2000000, F_CPU);
	XMEGACLK_SetCPUClockSource(CLOCK_SRC_PLL);
	XMEGACLK_StartInternalOscillator(CLOCK_SRC_INT_RC32MHZ);
	XMEGACLK_StartDFLL(CLOCK_SRC_INT_RC32MHZ, DFLL_REF_INT_USBSOF, F_USB);
	PMIC.CTRL = PMIC_LOLVLEN_bm | PMIC_MEDLVLEN_bm | PMIC_HILVLEN_bm;
#endif

  /* Disable JTAG if supported (frees PF4/PF5 on devices that have JTD) */
#ifdef JTD
  MCUCR |= (1 << JTD);
  MCUCR |= (1 << JTD);
#endif

    // --- LCD ---
  HD44780_Initialize();
  HD44780_WriteCommand(LCD_ON | CURSOR_NONE);
  HD44780_WriteCommand(LCD_CLEAR);
  HD44780_WriteCommand(LCD_HOME);
  HD44780_WriteCommand(LCD_INCR_RIGHT);

    // --- LEDs (PB7 réservé pour servo, pas LED3) ---
  DDRB  |= (1 << LED1) | (1 << LED2);
  DDRE  |= (1 << LED4);
  PORTB &= ~((1 << LED1) | (1 << LED2));
  PORTE &= ~(1 << LED4);

    // --- Boutons ---
  DDRB  &= ~(1 << BOUTON_G);
  DDRC  &= ~(1 << BOUTON_C);
  PORTB |=  (1 << BOUTON_G);
  PORTC |=  (1 << BOUTON_C);

    // --- Sonar ---
  sonar_init();

	/* USB Initialization */
	USB_Init();

    // --- Servo (utilise Timer1/OC1C sur PB7 pour éviter conflits avec LUFA) ---
    setupServo();
    setServoAngle(5);  // Position initiale
}

/** Event handler for the USB_Connect event. This indicates that the device is enumerating via the status LEDs. */
void EVENT_USB_Device_Connect(void)
{
}

/** Event handler for the USB_Disconnect event. This indicates that the device is no longer connected to a host via
 *  the status LEDs.
 */
void EVENT_USB_Device_Disconnect(void)
{
}

/** Event handler for the USB_ConfigurationChanged event. This is fired when the host set the current configuration
 *  of the USB device after enumeration - the device endpoints are configured.
 */
void EVENT_USB_Device_ConfigurationChanged(void)
{
  Endpoint_ConfigureEndpoint(ENDPOINT_DIR_IN | 1, EP_TYPE_INTERRUPT, 64, 1);
  Endpoint_ConfigureEndpoint(ENDPOINT_DIR_OUT | 2, EP_TYPE_INTERRUPT, 64, 1);
}

/** Event handler for the USB_ControlRequest event. This is used to catch and process control requests sent to
 *  the device from the USB host before passing along unhandled control requests to the library for processing
 *  internally.
 */
void EVENT_USB_Device_ControlRequest(void)
{
}
