#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <libusb-1.0/libusb.h>

#define VID 0x4242
#define PID 0x0001
#define INTERFACE 0
#define EP_OUT 0x02
#define EP_IN  0x81

#define TIMEOUT_MS 1000

static int send_angle(libusb_device_handle *h, int angle) {
    char buf[32];
    int len = snprintf(buf, sizeof(buf), "%d\n", angle);
    int transferred = 0;
    int rc = libusb_interrupt_transfer(h, EP_OUT, (unsigned char*)buf, len, &transferred, TIMEOUT_MS);
    if (rc != 0) return rc;
    return 0;
}

static int read_response(libusb_device_handle *h, char *outbuf, int outlen) {
    unsigned char buf[128];
    int transferred = 0;
    int rc = libusb_interrupt_transfer(h, EP_IN, buf, sizeof(buf), &transferred, TIMEOUT_MS);
    if (rc != 0) return rc;
    int n = transferred < outlen-1 ? transferred : outlen-1;
    memcpy(outbuf, buf, n);
    outbuf[n] = '\0';
    return 0;
}

static int parse_distance(const char *s) {
    // look for D:<digits>
    const char *p = strstr(s, "D:");
    if (!p) return -1;
    p += 2;
    // skip non-digits
    while (*p && (*p < '0' || *p > '9')) p++;
    if (!*p) return -1;
    int val = 0;
    while (*p && *p >= '0' && *p <= '9') {
        val = val*10 + (*p - '0');
        p++;
    }
    return val;
}

int main(int argc, char **argv) {
    const char *csv_path = "Minimal.csv"; // same folder
    if (argc > 1) csv_path = argv[1];

    libusb_context *ctx = NULL;
    libusb_device_handle *handle = NULL;
    int rc = libusb_init(&ctx);
    if (rc != 0) {
        fprintf(stderr, "libusb_init failed: %d\n", rc);
        return 1;
    }

    handle = libusb_open_device_with_vid_pid(ctx, VID, PID);
    if (!handle) {
        fprintf(stderr, "Device VID=0x%04x PID=0x%04x not found\n", VID, PID);
        libusb_exit(ctx);
        return 2;
    }

    if (libusb_kernel_driver_active(handle, INTERFACE) == 1) {
        libusb_detach_kernel_driver(handle, INTERFACE);
    }

    rc = libusb_claim_interface(handle, INTERFACE);
    if (rc != 0) {
        fprintf(stderr, "Cannot claim interface: %d\n", rc);
        libusb_close(handle);
        libusb_exit(ctx);
        return 3;
    }

    FILE *csv = fopen(csv_path, "a");
    if (!csv) {
        perror("fopen");
        libusb_release_interface(handle, INTERFACE);
        libusb_close(handle);
        libusb_exit(ctx);
        return 4;
    }

    // write header if file was empty
    fseek(csv, 0, SEEK_END);
    long end = ftell(csv);
    if (end == 0) {
        fprintf(csv, "timestamp,angle,distance_cm\n");
        fflush(csv);
    }

    int angles_start = 5;
    int angles_end = 19;

    for (int a = angles_start; a <= angles_end; ++a) {
        printf("Setting angle %d\n", a);
        rc = send_angle(handle, a);
        if (rc != 0) {
            fprintf(stderr, "send_angle failed: %d\n", rc);
            continue;
        }

        // wait a bit for settling
        usleep(200 * 1000);

        int retries = 5;
        int dist = -1;
        char resp[128];
        while (retries-- > 0) {
            rc = read_response(handle, resp, sizeof(resp));
            if (rc == 0) {
                printf("RX: %s\n", resp);
                dist = parse_distance(resp);
                if (dist >= 0) break; // got a number
                // if response has D:--- or no D, retry
            } else {
                fprintf(stderr, "read_response rc=%d\n", rc);
            }
            usleep(100 * 1000);
        }

        time_t now = time(NULL);
        if (dist >= 0) {
            fprintf(csv, "%ld,%d,%d\n", now, a, dist);
            fflush(csv);
            printf("Wrote %d,%d\n", a, dist);
        } else {
            fprintf(csv, "%ld,%d,\n", now, a);
            fflush(csv);
            printf("No valid distance for angle %d\n", a);
        }

        // small pause before next angle
        usleep(100 * 1000);
    }

    fclose(csv);
    libusb_release_interface(handle, INTERFACE);
    libusb_close(handle);
    libusb_exit(ctx);
    return 0;
}
