#!/usr/bin/env python3
"""
Host script to control the LUFA device (vendor device with interrupt endpoints)
and write angle,distance measurements to a CSV located next to Minimal.c.

Requirements:
- libusb installed (macOS: brew install libusb)
- pyusb Python package (pip install pyusb)

Usage:
  python3 read_usb_to_csv.py

Behavior:
- Finds device with VID=0x4242 PID=0x0001
- Claims interface 0 and uses endpoints 0x02 (OUT) and 0x81 (IN)
- Iterates angles from 5 to 19, for each angle:
    - sends ASCII angle (e.g. "10\n") to OUT endpoint
    - waits a short settling delay
    - does an interrupt read from IN endpoint
    - parses response of form "A:<angle> D:<dist>\n" and appends to CSV
- Continues looping until Ctrl-C
"""

import usb.core
import usb.util
import time
import csv
import os

OUT_EP = None
IN_EP = None
INTERFACE = None
CSV_NAME = os.path.join(os.path.dirname(__file__), 'Minimal.csv')

SETTLE_MS = 200
TIMEOUT_MS = 1000


def find_device():
    # Auto-detect a device that exposes one interrupt IN endpoint and one interrupt OUT endpoint
    for dev in usb.core.find(find_all=True):
        try:
            for cfg in dev:
                for intf in cfg:
                    in_ep = None
                    out_ep = None
                    for ep in intf:
                        # transfer type in bmAttributes low 2 bits: 3 == interrupt
                        ttype = ep.bmAttributes & 0x03
                        if ttype != 0x03:
                            continue
                        if ep.bEndpointAddress & 0x80:
                            in_ep = ep.bEndpointAddress
                        else:
                            out_ep = ep.bEndpointAddress
                    if in_ep and out_ep:
                        # return device and interface number and endpoints
                        return dev, intf.bInterfaceNumber, in_ep, out_ep
        except Exception:
            continue
    raise RuntimeError("No suitable USB device with interrupt IN/OUT endpoints found")


def claim_device(dev, interface):
    try:
        dev.set_configuration()
    except usb.core.USBError:
        pass
    cfg = dev.get_active_configuration()
    intf = usb.util.find_descriptor(cfg, bInterfaceNumber=interface)
    if dev.is_kernel_driver_active(interface):
        try:
            dev.detach_kernel_driver(interface)
        except Exception as e:
            raise RuntimeError(f"Could not detach kernel driver: {e}")
    usb.util.claim_interface(dev, interface)
    return intf


def release_device(dev):
    try:
        usb.util.release_interface(dev, INTERFACE)
    except Exception:
        pass
    try:
        dev.attach_kernel_driver(INTERFACE)
    except Exception:
        pass


def send_angle(dev, out_ep, angle):
    s = f"{angle}\n".encode('ascii')
    # write uses endpoint address; pyusb will use default configuration/interface
    dev.write(out_ep, s, TIMEOUT_MS)


def read_response(dev, in_ep):
    try:
        data = dev.read(in_ep, 64, TIMEOUT_MS)
        txt = bytes(data).split(b"\x00", 1)[0].decode('ascii', errors='ignore')
        return txt.strip()
    except usb.core.USBError:
        return None


def parse_resp(txt):
    # expect A:<angle> D:<dist>
    if not txt:
        return None, None
    try:
        parts = txt.replace(',', ' ').split()
        angle = None
        dist = None
        for p in parts:
            if p.startswith('A:'):
                angle = int(p[2:])
            if p.startswith('D:'):
                # strip trailing non-digits
                num = ''.join(c for c in p[2:] if (c.isdigit()))
                if num:
                    dist = int(num)
        return angle, dist
    except Exception:
        return None, None


def main():
    dev, interface, in_ep, out_ep = find_device()
    print(f"Found device: vid=0x{dev.idVendor:04x} pid=0x{dev.idProduct:04x} interface={interface} in_ep=0x{in_ep:02x} out_ep=0x{out_ep:02x}")
    claim_device(dev, interface)
    print("Device configured and interface claimed")

    # Ensure CSV exists and has header
    first_time = not os.path.exists(CSV_NAME)
    csvf = open(CSV_NAME, 'a', newline='')
    writer = csv.writer(csvf)
    if first_time:
        writer.writerow(['timestamp', 'angle', 'distance_cm'])
        csvf.flush()

    try:
        angles = list(range(5, 20))
        idx = 0
        direction = 1
        while True:
            ang = angles[idx]
            print(f"Setting angle {ang}")
            send_angle(dev, out_ep, ang)
            time.sleep(SETTLE_MS/1000.0)

            resp = read_response(dev, in_ep)
            if resp is None:
                print("No response (timeout)")
                # still write a line indicating no measurement
                writer.writerow([time.time(), ang, None])
                csvf.flush()
            else:
                print(f"RX: {resp}")
                a, d = parse_resp(resp)
                writer.writerow([time.time(), a if a is not None else ang, d])
                csvf.flush()

            # next angle
            idx += direction
            if idx >= len(angles):
                idx = len(angles)-1
                direction = -1
            elif idx < 0:
                idx = 0
                direction = 1

            # small pause between steps
            time.sleep(0.05)

    except KeyboardInterrupt:
        print('\nStopping, releasing device')
    finally:
        csvf.close()
        release_device(dev)


if __name__ == '__main__':
    main()
