#define F_CPU 16000000UL
#include <avr/io.h>
#include <util/delay.h>
#include <stddef.h>
#include <stdio.h>
#include "lcd/HD44780.h"
#include "servo.h"
#include "sonar.h"
#include <string.h>

#define LED1        1    // PB1
#define LED2        2    // PB2
#define LED3        7    // PB7
#define LED4        6    // PE7
#define BOUTON_G    3    // PB3
#define BOUTON_C    7    // PC7
#define DEBOUNCE_MS 20

// Configuration pour balayage complet (mappé 5..19 -> 0.5ms..2.4ms)
#define ANGLE_STEP      1
#define DELAY_BETWEEN   300   
#define ANGLE_START     5
#define ANGLE_END       19

int main(void) {
    // --- Désactivation JTAG ---
    MCUCR |= (1 << JTD);
    MCUCR |= (1 << JTD);

    // --- LCD ---
    HD44780_Initialize();
    HD44780_WriteCommand(LCD_ON | CURSOR_NONE);
    HD44780_WriteCommand(LCD_CLEAR);
    HD44780_WriteCommand(LCD_HOME);
    HD44780_WriteCommand(LCD_INCR_RIGHT);

    // --- LEDs (PB7 ignoré car Servo) ---
    DDRB  |= (1 << LED1) | (1 << LED2); 
    DDRE  |= (1 << LED4);
    PORTB &= ~((1 << LED1) | (1 << LED2));
    PORTE &= ~(1 << LED4);

    // --- Boutons ---
    DDRB  &= ~(1 << BOUTON_G);
    DDRC  &= ~(1 << BOUTON_C);
    PORTB |=  (1 << BOUTON_G);
    PORTC |=  (1 << BOUTON_C);

    // --- Servo ---
    setupServo();
    int16_t currentAngle = ANGLE_START;
    int8_t direction = ANGLE_STEP; 
    setServoAngle(currentAngle);

    // --- Sonar ---
    sonar_init();

    uint8_t autoMode = 0;
    uint8_t lastC = 1;
    uint8_t lastG = 1;
    char buffer[32];

    HD44780_DisplayLine(0, "Pret! Btn C=Auto");

    while (1) {
        uint8_t curC = (PINC & (1 << BOUTON_C)) ? 1 : 0;
        uint8_t curG = (PINB & (1 << BOUTON_G)) ? 1 : 0;

        // --- MANUEL ---
        if (lastG == 1 && curG == 0) {
            _delay_ms(DEBOUNCE_MS);
            if (!(PINB & (1 << BOUTON_G))) {
                autoMode = 0;
                
                currentAngle += 1;
                if (currentAngle > ANGLE_END) currentAngle = ANGLE_START;
                
                setServoAngle(currentAngle);
                _delay_ms(200);

                uint16_t dist = sonar_mesure();
                
                snprintf(buffer, sizeof(buffer), "Man A:%2d D:%3ucm  ", currentAngle, dist);
                HD44780_WriteCommand(LCD_HOME);
                HD44780_DisplayLine(0, buffer);
            }
        }

        // --- AUTO TOGGLE ---
        if (lastC == 1 && curC == 0) {
            _delay_ms(DEBOUNCE_MS);
            if (!(PINC & (1 << BOUTON_C))) {
                autoMode = !autoMode;
                if (autoMode) {
                    currentAngle = ANGLE_START;
                    direction = ANGLE_STEP;
                    HD44780_WriteCommand(LCD_HOME);
                    // 16 chars + null
                    HD44780_DisplayLine(0, "Mode Auto ON    ");
                    _delay_ms(2000);
                } else {
                    HD44780_WriteCommand(LCD_HOME);
                    HD44780_DisplayLine(0, "Mode Auto OFF   ");
                    _delay_ms(2000);
                }
            }
        }

        // --- AUTO LOOP ---
        if (autoMode) {
            // 1. Déplacer
            setServoAngle(currentAngle);
            
            // 2. Attendre stabilisation (court)
            _delay_ms(50);

            // 3. Mesurer
            uint16_t dist = sonar_mesure();

            // 4. Afficher
            if (dist < 400)
                snprintf(buffer, sizeof(buffer), "A:%2d D:%3ucm      ", currentAngle, dist);
            else
                snprintf(buffer, sizeof(buffer), "A:%2d D:---        ", currentAngle);
                
            HD44780_WriteCommand(LCD_HOME);
            HD44780_DisplayLine(0, buffer);

            // 5. Attente avant prochain pas
            _delay_ms(DELAY_BETWEEN);

            // 6. Calcul suivant
            currentAngle += direction;

            if (currentAngle >= ANGLE_END) {
                currentAngle = ANGLE_END;
                direction = -ANGLE_STEP;
            } 
            else if (currentAngle <= ANGLE_START) {
                currentAngle = ANGLE_START;
                direction = ANGLE_STEP;
            }
        }

        lastC = curC;
        lastG = curG;
    }
}
