#include "servo.h"
#include <util/delay.h>

/*
 * Servo PWM using Timer1 Channel C -> PB7 (OC1C)
 * This avoids conflicts with LUFA which may use Timer0.
 * 
 * F_CPU = 16 MHz, Prescaler = 8 => Tick = 0.5 us
 * For 50Hz servo: Period = 20ms = 40000 ticks
 * Pulse 1ms = 2000 ticks, Pulse 2ms = 4000 ticks
 */

#define ICR1_TOP      40000   // 20ms period for 50Hz
#define PULSE_MIN     2000    // 1.0ms = position min
#define PULSE_MAX     4000    // 2.0ms = position max

// Variables globales
uint32_t angle = ANGLE_MIN;
int servoDirection = 1;

// Mapping de l'angle (5..19) vers pulse (2000..4000 ticks)
static uint16_t angle_to_pulse(uint32_t a) {
    if (a < ANGLE_MIN) a = ANGLE_MIN;
    if (a > ANGLE_MAX) a = ANGLE_MAX;
    
    // Interpolation linéaire: 5 -> 2000, 19 -> 4000
    uint32_t range_in = ANGLE_MAX - ANGLE_MIN;   // 14
    uint32_t range_out = PULSE_MAX - PULSE_MIN;  // 2000
    
    return (uint16_t)(PULSE_MIN + ((a - ANGLE_MIN) * range_out) / range_in);
}

void setupServo(void) {
    // PB7 comme sortie (OC1C)
    DDRB |= (1 << PB7);
    
    // Reset Timer1
    TCCR1A = 0;
    TCCR1B = 0;
    TCCR1C = 0;
    TCNT1 = 0;
    
    // TOP = ICR1 pour période 50Hz
    ICR1 = ICR1_TOP;
    
    // Mode 14: Fast PWM, TOP = ICR1
    // WGM13:0 = 1110 => WGM13=1, WGM12=1, WGM11=1, WGM10=0
    // COM1C1=1: Clear OC1C on Compare Match (non-inverting)
    TCCR1A = (1 << COM1C1) | (1 << WGM11);
    TCCR1B = (1 << WGM13) | (1 << WGM12) | (1 << CS11); // Prescaler 8
    
    // Position initiale
    OCR1C = angle_to_pulse(ANGLE_MIN);
    angle = ANGLE_MIN;
}

void setServoAngle(uint32_t newAngle) {
    if (newAngle < ANGLE_MIN) newAngle = ANGLE_MIN;
    if (newAngle > ANGLE_MAX) newAngle = ANGLE_MAX;
    
    OCR1C = angle_to_pulse(newAngle);
    angle = newAngle;
}

uint32_t getAngle(void) {
    return angle;
}

int getServoDirection(void) {
    return servoDirection;
}

void setServoDirection(int newDirection) {
    servoDirection = newDirection;
}

void rotateServo(void) {
    if (angle >= ANGLE_MAX) servoDirection = -1;
    else if (angle <= ANGLE_MIN) servoDirection = 1;
    
    setServoAngle(angle + servoDirection);
    _delay_ms(36);
}
