#!/usr/bin/env python3
"""
Sonar Visualization Interface - PYTHON CONTROLLED SERVO
--------------------------------------------------------
Real-time polar plot of sonar data.
Python controls the servo angle and reads distance measurements.

Usage:
  python3 sonar_gui.py

Dependencies:
  - pyusb
  - matplotlib
  - numpy
"""

import sys
import usb.core
import usb.util
import numpy as np
import matplotlib.pyplot as plt
from matplotlib.animation import FuncAnimation
import time

# --- Configuration ---
VID = 0x4242
PID = 0x0001
ANGLE_MIN = 5
ANGLE_MAX = 19
TIMEOUT_MS = 1000
SWEEP_DELAY = 0.3  # Délai entre chaque angle (comme v0: 300ms)

# --- USB Globals ---
dev = None
ep_in = None
ep_out = None
interface_num = None

# --- Servo Control State ---
current_angle = ANGLE_MIN
servo_direction = 1  # 1 = increasing, -1 = decreasing

def init_usb():
    global dev, ep_in, ep_out, interface_num
    
    dev = usb.core.find(idVendor=VID, idProduct=PID)
    if dev is None:
        raise ValueError(f"Device not found (VID=0x{VID:04x}, PID=0x{PID:04x}).\n"
                         "Make sure the firmware is flashed and the device is connected.")

    print(f"Device found: {dev.product} (VID=0x{dev.idVendor:04x})")
    
    # Find interface and endpoints
    found_intf = None
    e_in = None
    e_out = None
    
    try:
        for cfg in dev:
            for intf in cfg:
                for ep in intf:
                    if (ep.bmAttributes & 0x03) == 0x03:  # Interrupt
                        if ep.bEndpointAddress & 0x80:   # IN
                            e_in = ep
                        else:                             # OUT
                            e_out = ep
                        found_intf = intf
                if found_intf: break
            if found_intf: break
    except Exception as e:
        print(f"Error iterating descriptors: {e}")

    if not found_intf or not e_in:
        raise ValueError("Could not find Interrupt IN endpoint.")

    interface_num = found_intf.bInterfaceNumber
    ep_in = e_in
    ep_out = e_out
    
    print(f"Using Interface {interface_num} (IN: 0x{ep_in.bEndpointAddress:02x})")
    if ep_out:
        print(f"  OUT endpoint: 0x{ep_out.bEndpointAddress:02x}")
    else:
        print("  No OUT endpoint found - will use control transfers")

    # Detach kernel driver if needed
    try:
        if dev.is_kernel_driver_active(interface_num):
            dev.detach_kernel_driver(interface_num)
    except Exception:
        pass

    # Set config and claim
    try:
        dev.set_configuration()
    except:
        pass
    try:
        usb.util.claim_interface(dev, interface_num)
    except Exception as e:
        print(f"Warning: Could not claim interface: {e}")

def send_angle(angle):
    """Send angle command to the board."""
    global ep_out, dev
    
    if ep_out is None:
        return False
    
    try:
        # Format: "S:XX" where XX is the angle
        cmd = f"S:{angle}\n".encode('ascii')
        ep_out.write(cmd, TIMEOUT_MS)
        return True
    except usb.core.USBError as e:
        # Silently ignore write errors
        return False

def read_data():
    """Read one data packet from the board. Returns (angle, distance) or (None, None)."""
    if ep_in is None:
        return None, None
    try:
        data = ep_in.read(64, TIMEOUT_MS)
        text = bytes(data).split(b'\x00')[0].decode('ascii', errors='ignore').strip()
        # Parse "A:X D:Y"
        parts = text.split()
        a_val = None
        d_val = None
        for p in parts:
            if p.startswith("A:"):
                try:
                    a_val = int(p.split(':')[1])
                except:
                    pass
            elif p.startswith("D:"):
                try:
                    num = ''.join(c for c in p.split(':')[1] if c.isdigit())
                    if num:
                        d_val = int(num)
                except:
                    pass
        return a_val, d_val
    except usb.core.USBTimeoutError:
        return None, None
    except usb.core.USBError as e:
        print(f"USB Error: {e}")
        return None, None

# --- Visualization ---
angles = list(range(ANGLE_MIN, ANGLE_MAX + 1))
distances = [0.0] * len(angles)

# Map angles 5..19 to radians 0..pi (180 degrees sweep)
theta_map = np.linspace(0, np.pi, len(angles))

# Setup Plot
plt.style.use('dark_background')
fig = plt.figure(figsize=(10, 10))
ax = fig.add_subplot(111, projection='polar')

# Visual elements
line, = ax.plot([], [], color='#00ff88', linewidth=2, alpha=0.8)
scat = ax.scatter([], [], c='#ff3366', s=80, marker='o', alpha=0.9, zorder=5)

# Sweep line indicator
sweep_line, = ax.plot([], [], color='#ffff00', linewidth=3, alpha=0.7)

# Config Appearance
ax.set_ylim(0, 200)
ax.set_title("🎯 Sonar Real-Time Scan (Python Controlled)", fontsize=16, color='white', pad=20, fontweight='bold')
ax.grid(True, alpha=0.3, color='#00ff88')
ax.set_theta_zero_location("N")
ax.set_theta_direction(-1)
ax.set_thetamin(0)
ax.set_thetamax(180)
ax.set_facecolor('#0a0a0a')
fig.patch.set_facecolor('#0a0a0a')

# Status text
status_text = ax.text(0.5, -0.1, "Initializing...", transform=ax.transAxes, 
                       ha='center', fontsize=12, color='#888888')

last_update_time = time.time()

def update(frame):
    global current_angle, servo_direction, last_update_time
    
    now = time.time()
    
    # Control sweep timing (like v0: 300ms delay)
    if now - last_update_time >= SWEEP_DELAY:
        last_update_time = now
        
        # Send current angle to board
        send_angle(current_angle)
        
        # Read data from board
        a, d = read_data()
        
        if a is not None and d is not None:
            # Update the distance array for this angle
            try:
                idx = angles.index(a)
                if d > 400:
                    d = 400  # Cap at max range
                distances[idx] = d
                status_text.set_text(f"Angle: {a}° | Distance: {d} cm")
            except ValueError:
                pass  # angle out of range
        else:
            status_text.set_text(f"Angle: {current_angle}° | Waiting...")
        
        # Update angle for next iteration (like v0: sweep 5-19)
        current_angle += servo_direction
        if current_angle >= ANGLE_MAX:
            current_angle = ANGLE_MAX
            servo_direction = -1
        elif current_angle <= ANGLE_MIN:
            current_angle = ANGLE_MIN
            servo_direction = 1
    
    # Build plot data from all stored distances
    valid_data = [(theta_map[i], distances[i]) for i in range(len(angles)) if distances[i] > 0]
    
    if valid_data:
        ts, rs = zip(*valid_data)
        line.set_data(ts, rs)
        scat.set_offsets(np.c_[ts, rs])
    
    # Update sweep indicator
    try:
        idx = angles.index(current_angle)
        sweep_theta = theta_map[idx]
        sweep_line.set_data([sweep_theta, sweep_theta], [0, 200])
    except ValueError:
        pass
    
    return line, scat, status_text, sweep_line

def on_close(event):
    print("\nClosing...")

fig.canvas.mpl_connect('close_event', on_close)

def main():
    try:
        init_usb()
    except Exception as e:
        print(f"USB Error: {e}")
        print("\nRunning in DEMO MODE (No USB Device)")
    
    print("Starting visualization... (Python controls servo)")
    print(f"Sweep delay: {SWEEP_DELAY}s (like v0: 300ms)")
    print("Press Ctrl+C or close window to exit.")
    
    # Animation interval is 50ms, but sweep is controlled by SWEEP_DELAY
    ani = FuncAnimation(fig, update, frames=None, interval=50, blit=False, cache_frame_data=False)
    
    plt.tight_layout()
    plt.show()

if __name__ == "__main__":
    main()
