#include "cli.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>     
#include <unistd.h>
#include <libusb.h>

#include "clipboard.h"
#include "../constants.h"
#include "usb.h"

static char line[256];

int cli(const BiniouKey* binioukey, const libusb_context *context) {
    int debug_mode = 1;

    printf("(type help for available commands)\n");

    while (1) {

        fflush(stderr);
        printf("> ");
        fflush(stdout);

        if (!fgets(line, sizeof(line), stdin))
            continue;

        // Retirer le \n
        line[strcspn(line, "\n")] = '\0';

        // Skip si vide
        if (strlen(line) == 0)
            continue;

        // Découpe des arguments saisis
        char *cmd = strtok(line, " ");
        char *arg1 = strtok(NULL, " ");
        char *arg2 = strtok(NULL, " ");

        // exit
        if (strcmp(cmd, "exit") == 0) {
            printf("[OK] Sarrasin Defender bat en retraite !\n");
            return EXIT_SUCCESS;
        }

        // debug <on|off|(empty)> : mode débug pour les devs (nous)
        if (strcmp(cmd, "debug") == 0) {
            if (arg1 == NULL) {
                debug_mode = debug_mode == 0 ? 1 : 0;
            }

            else if (strcmp(arg1, "on") == 0) {
                debug_mode = 0;
            }

            else {
                debug_mode = 1;
            }

            if (debug_mode == 0) {
                libusb_set_option(context, LIBUSB_OPTION_LOG_LEVEL, LIBUSB_LOG_LEVEL_DEBUG);

            } else {
                libusb_set_option(context, LIBUSB_OPTION_LOG_LEVEL, LIBUSB_LOG_LEVEL_ERROR);
            }

            printf("[DEBUG] Debug mode is : %s\n", debug_mode == 0 ? "activated" : "deactivated");

            continue;
        }

        // encrypt <file> <passphrase>
        if (strcmp(cmd, "encrypt") == 0) {
            // Générer la commande à envoyer à la binioukey

            const char *passphrase = get_key(binioukey, debug_mode, 0);

            if (passphrase == NULL) {
                continue;
            }
                       
            biniou_encrypt(arg1, passphrase);
            free(passphrase);

            continue;
        }

        // decrypt <file> <key_id>
        if (strcmp(cmd, "decrypt") == 0) {

            const char *passphrase = get_key(binioukey, debug_mode, 0);

            biniou_decrypt(arg1, passphrase);
            free(passphrase);

            continue;
        }

        if (strcmp(cmd, "generate") == 0) {

            const int status = create_key(binioukey);

            if (status == 0) {
                printf("Key generation has been acknowledged by your BiniouKey\n");
            }

            continue;
        }

        // copy [--sanitize]
        if (strcmp(cmd, "copy") == 0) {
            const int sanitise = arg1 != NULL && strcmp(arg1, "--sanitize") == 0 ? 0 : 1;

            const char *passphrase = get_key(binioukey, debug_mode, sanitise);

            copy_to_clipboard(passphrase);
            free(passphrase);
            continue;
        }

        if (strcmp(cmd, "help") == 0) {
            printf("Available commands:\n\thelp : this\n\tencrypt <file> : encrypts the specified file\n\tdecrypt <file> : decrypts the specified .gpg file\n\tgenerate : asks your BiniouKey to generate a password\n\tcopy [--sanitize] : copy a passphrase to clipboard, optionally sanitized to be used with gpg\n\texit : exits Sarazin Defender\n\tdebug [on|off] : turns on/off debug mode\n\n");
            continue;
        }

        // commande inconnue
        printf("Unknown command: %s\n", cmd);
        printf("Available commands:\n\thelp\n\tencrypt <file>\n\tdecrypt <file>\n\tgenerate\n\tcopy [--sanitize]\n\texit\n\tdebug [on|off]\n\n");
    }
}

int create_key(const BiniouKey* binioukey) {
    // setup des variables
    const uint8_t request_type = LIBUSB_REQUEST_TYPE_VENDOR | LIBUSB_RECIPIENT_DEVICE | LIBUSB_ENDPOINT_OUT;
    const uint8_t request = CONTROL_ENDPOINT_CMD_GENERATE;
    const uint16_t value = 0;
    const uint16_t index = 0;
    unsigned char *data = NULL;
    const uint16_t length = 0;
    const unsigned int timeout = 5000;

    // envoie du truc de transfert
    const int status = libusb_control_transfer(
        binioukey->handle,
        request_type,
        request,
        value,
        index,
        data,
        length,
        timeout
    );

    if (status < 0) {
        fprintf(stderr, "[KO] Control transfer failed: %s\n", libusb_error_name(status));
    }

    return status;
}

int biniou_encrypt(const char *filepath, const char *passphrase) {
    if (!filepath  || !passphrase) {
        printf("Usage: encrypt <file>\n");
        return -1;
    }

    char cmd[512];
    snprintf(cmd, sizeof(cmd),
             "gpg --batch --yes --passphrase \"%s\" --symmetric \"%s\"",
             passphrase, filepath);

    const int ret = system(cmd);
    if (ret != 0) {
        printf("[KO] gpg encryption failed (code %d)\n", ret);
        return -1;
    }

    printf("[OK] Fichier encrypté : %s.gpg\n", filepath);
    return 0;
}

int biniou_decrypt(const char *filepath, const char* passphrase) {
    const size_t len = strlen(filepath);

    // Vérifier que le fichier se termine par ".gpg"
    if (len < 4 || strcmp(filepath + len - 4, ".gpg") != 0) {
        printf("[KO] Le fichier n'a pas l'extension .gpg : %s\n", filepath);
        return -1;
    }

    // Construire le nom du fichier de sortie (sans .gpg)
    const size_t out_len = len - 4 + 1;
    char *output_path = malloc(out_len);
    if (!output_path) return -1;

    strncpy(output_path, filepath, len - 4);
    output_path[len - 4] = '\0';

    // Construire la commande GPG
    char cmd[512];
    snprintf(cmd, sizeof(cmd),
             "gpg --batch --yes --passphrase \"%s\" --output \"%s\" --decrypt \"%s\"",
             passphrase, output_path, filepath);

    const int ret = system(cmd);
    if (ret != 0) {
        printf("[KO] gpg decryption failed\n");
        free(output_path);
        return -1;
    }

    printf("[OK] Fichier décrypté : %s\n", filepath);
    return 0;
}

char* get_key(const BiniouKey* binioukey, const int debug_mode, const int sanitise) {

    const int timeout_in = 30000;
    int transfered;

    unsigned char data[BINIOUKEY_MAX_KEY_SIZE + 1]; // size passphrase + \0

    printf("Select a key from your BiniouKey and press CENTRE BUTTON\n");

    const int status = libusb_interrupt_transfer(
        binioukey->handle, VENDOR_IN_EPADDR,
        data, BINIOUKEY_MAX_KEY_SIZE, &transfered, timeout_in
    );

    if (status != 0) {
        perror("[KO] Couldn't read data from the Binioukey\n");
        return NULL;
    }

    if (transfered != BINIOUKEY_MAX_KEY_SIZE) {
        perror("[KO] Not enough bytes were received");
        return NULL;
    }

    data[transfered] = '\0';

    if (debug_mode == 0) {
        printf("[OK] Passphrase received from the Binioukey: %s\n", data);
    }


     // Allocation dynamique de la chaîne échappée
    if (sanitise == 0) {
        const size_t max = 2 * strlen((char*)data) + 1;
        char *escaped = malloc(max);

        sanitize_passphrase(escaped, (char*)data);
        return escaped; // L'appelant devra free()
    }
    // Cas où on ne sanitise pas : on renvoie quand même une chaîne allouée avec malloc
    // Comme ça on peut free() dans tous les cas et c'est safe!
    else {
        char* no_escape = malloc(sizeof(char) * (transfered + 1));
        memcpy(no_escape, data, transfered + 1);
        return no_escape;
    }
}

void sanitize_passphrase(char *dst, const char *src) {
    while (*src) {
        if (*src == '\\' || *src == '"' || *src == '`' || *src == '$')
            *dst++ = '\\';
        *dst++ = *src++;
    }
    *dst = '\0';
}
