//
// Created by léo on 25/10/2025.
//

#include "usb.h"
#include "../constants.h"
#include <stdio.h>
#include <stdlib.h>

BiniouKey* waitForDevice(libusb_context *context) {

    libusb_device **devices;
    BiniouKey* binioukey = NULL;

    do {
        // On va chercher tous les périphériques
        const ssize_t list_size = libusb_get_device_list(context, &devices);
        if (list_size < 0) {
            printf("Cannot read USB devices\n");
            return NULL;
        }
        if (list_size == 0) {
            printf("No devices to read\n");
            return NULL;
        }
        for (int i = 0; i < list_size; i++) {
            // On essaye de récupérer le descripteur
            libusb_device *device = devices[i];
            struct libusb_device_descriptor descriptor;
            const int descriptor_get_status = libusb_get_device_descriptor(device, &descriptor);
            // Pas réussi
            if (descriptor_get_status != LIBUSB_SUCCESS) {
                continue;
            }
            // Sinon on vérifie que c'est une Binioukey
            if (descriptor.idVendor == VENDOR_ID && descriptor.idProduct == PRODUCT_ID) {
                printf("Your Binioukey is registered and ready to be used! 🎉\n");
                binioukey = malloc(sizeof(BiniouKey));
                libusb_ref_device(device);
                binioukey->device = device;
                binioukey->descriptor = malloc(sizeof(struct libusb_device_descriptor));
                *binioukey->descriptor = descriptor;

                // Maintenant on essaye de récupérer la configuration
                int status = getConfigDescriptor(binioukey);
                // Si on arrive pas tant pis hein voilà
                if (status == EXIT_FAILURE) {
                    printf("Cannot get configuration : failing");
                    binioukey = NULL;
                    continue;
                }

                // Et enfin on le handler
                status = getDeviceHandle(binioukey);
                // Pareil
                if (status == EXIT_FAILURE) {
                    printf("Cannot get device handler : failing");
                    binioukey = NULL;
                }
            }
        }
        libusb_free_device_list(devices, 1);

        if (binioukey == NULL) {
            printf("Unable to find any BiniouKey! Press Enter when you want to try again\n");
            getchar();
        }

    } while (binioukey == NULL);

    return binioukey;
}

int getConfigDescriptor(BiniouKey *binioukey) {
    struct libusb_config_descriptor *configdesc;
    const int status = libusb_get_active_config_descriptor(binioukey->device, &configdesc);
    if (status != 0) {
        perror("libusb_get_active_config_descriptor");
        return EXIT_FAILURE;
    }
    binioukey->configuration = configdesc;
    return EXIT_SUCCESS;
}

int getDeviceHandle(BiniouKey *binioukey) {
    libusb_device_handle *handle;
    int status = libusb_open(binioukey->device, &handle);
    if (status != 0) {
        perror("libusb_open");
        return EXIT_FAILURE;
    }
    binioukey->handle = handle;

    // Set the configuration explicitly
    status = libusb_set_configuration(handle, binioukey->configuration->bConfigurationValue);
    if (status != 0) {
        perror("libusb_set_configuration");
        libusb_close(handle);
        return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}

void closeAndExit(libusb_context* context, const BiniouKey *binioukey) {
    if (binioukey != NULL) {
        libusb_close(binioukey->handle);
        libusb_free_config_descriptor(binioukey->configuration);
        libusb_unref_device(binioukey->device);
        free(binioukey->descriptor);
    }
    free((void*)binioukey);
    libusb_exit(context);
}

void printEndpoints(const BiniouKey *biniouKey) {
    printf("Listing all endpoints:\n");
    printf("Number of interfaces: %d\n", biniouKey->configuration->bNumInterfaces);

    for (int i = 0; i < biniouKey->configuration->bNumInterfaces; i++) {
        printf("\nInterface %d:\n", i);
        const struct libusb_interface *iface = &biniouKey->configuration->interface[i];

        printf("  Number of alternate settings: %d\n", iface->num_altsetting);

        for (int j = 0; j < iface->num_altsetting; j++) {
            const struct libusb_interface_descriptor *altsetting = &iface->altsetting[j];
            printf("  Alternate setting %d:\n", j);
            printf("    Number of endpoints: %d\n", altsetting->bNumEndpoints);

            for (int k = 0; k < altsetting->bNumEndpoints; k++) {
                const struct libusb_endpoint_descriptor *ep = &altsetting->endpoint[k];

                const char *direction = (ep->bEndpointAddress & LIBUSB_ENDPOINT_DIR_MASK) ? "IN" : "OUT";
                const char *type;
                switch (ep->bmAttributes & LIBUSB_TRANSFER_TYPE_MASK) {
                    case LIBUSB_TRANSFER_TYPE_CONTROL: type = "Control";
                        break;
                    case LIBUSB_TRANSFER_TYPE_ISOCHRONOUS: type = "Isochronous";
                        break;
                    case LIBUSB_TRANSFER_TYPE_BULK: type = "Bulk";
                        break;
                    case LIBUSB_TRANSFER_TYPE_INTERRUPT: type = "Interrupt";
                        break;
                    default: type = "Unknown";
                        break;
                }

                printf("      Endpoint %d: address=0x%02X, direction=%s, type=%s, max_packet_size=%d\n",
                       k, ep->bEndpointAddress, direction, type, ep->wMaxPacketSize);
            }
        }
    }
}
