
#include "Minimal.h"
#include "Descriptors.h"
#include <avr/wdt.h>
#include "lcd/lcd.h"
#include "leds/leds.h"
#include "boutons/bouton_general.h"
#include "pir/pir.h"
#include "sound/sound.h"
#include <string.h>

#define CMD_BUFFER_SIZE 32
#define NB_ROOMS 4
#define PIR_TIMEOUT_MS 5000  // 5 secondes
static char buffer[CMD_BUFFER_SIZE];
static uint8_t index = 0;

static uint8_t system_on = 0;
static uint8_t last_button_state = 0;

static uint8_t ch_state[4] = {0, 0, 0, 0}; // CH1..CH4
static uint8_t force_message = 0;

typedef struct {
    uint8_t state; // 0 = no motion, 1 = motion detected
    uint32_t last_motion_time; // timestamp of last motion
} Room;

static Room rooms[NB_ROOMS] = {0};
static uint32_t system_time_ms = 0;


USB_ClassInfo_CDC_Device_t VirtualSerial_CDC_Interface =
{
    .Config =
    {
        .ControlInterfaceNumber         = INTERFACE_ID_CDC_CCI,
        .DataINEndpoint                 =
        {
            .Address                = CDC_TX_EPADDR,
            .Size                   = CDC_TXRX_EPSIZE,
            .Banks                  = 1,
        },
        .DataOUTEndpoint                =
        {
            .Address                = CDC_RX_EPADDR,
            .Size                   = CDC_TXRX_EPSIZE,
            .Banks                  = 1,
        },
        .NotificationEndpoint           =
        {
            .Address                = CDC_NOTIFICATION_EPADDR,
            .Size                   = CDC_NOTIFICATION_EPSIZE,
            .Banks                  = 1,
        },
    },
};

void SetupHardware(void)
{
    MCUSR &= ~(1 << WDRF);
    wdt_disable();
    USB_Init();
    //clock_prescale_set(clock_div_1);
    lcd_init();
    lcd_clear();
    lcd_write_string("USB READY");

    leds_init();
    leds_all_off();

    bouton_general_init();
    pir_init();
    sound_init();
   
    
}

void process_command(char* cmd);
void lcd_show_status(void);
void handle_pirs(void);
void handle_timeouts(void);

void lcd_show_message(const char* msg)

{
    lcd_clear();
    lcd_set_cursor(0, 0);
    lcd_write_string(msg);
}


int main(void)
{
    SetupHardware();
    GlobalInterruptEnable();

    for (;;)
    {
       //  Lecture bouton
        uint8_t btn = bouton_general_is_pressed();

        if (btn && !last_button_state)
        {
            system_on = !system_on;

            if (system_on)
            {
                leds_all_on();
                lcd_clear();
                lcd_write_string("SYSTEM ON");
            }
            else
            {
                leds_all_off();
                lcd_clear();
                lcd_write_string("SYSTEM OFF");
            }

            _delay_ms(200); // anti-rebond simple
        }

        last_button_state = btn;
        _delay_ms(10);
        system_time_ms += 10;

        CDC_Task();
        USB_USBTask();
        handle_pirs();
        handle_timeouts();
    }
}


void EVENT_USB_Device_ConfigurationChanged(void)
{
    CDC_Device_ConfigureEndpoints(&VirtualSerial_CDC_Interface);
}

void EVENT_USB_Device_ControlRequest(void)
{
    CDC_Device_ProcessControlRequest(&VirtualSerial_CDC_Interface);
}


void CDC_Task(void)
{
    int16_t ReceivedByte = CDC_Device_ReceiveByte(&VirtualSerial_CDC_Interface);

    if (ReceivedByte >= 0)
    {
        char c = (char)ReceivedByte;

        if (c == '\r' || c == '\n')
        {
            buffer[index] = '\0';
            process_command(buffer);
            index = 0;

            // Accusé de réception vers Python
            CDC_Device_SendString(&VirtualSerial_CDC_Interface, "OK\n");
        }
        else if (index < CMD_BUFFER_SIZE - 1)
        {
            buffer[index++] = c;
        }
    }

    CDC_Device_USBTask(&VirtualSerial_CDC_Interface);
}


void process_command(char* cmd)
{
    if (!strcmp(cmd, "CH1_ON"))  { led_on('B', LED1); ch_state[0] = 1; }
    else if (!strcmp(cmd, "CH1_OFF")) { led_off('B', LED1); ch_state[0] = 0; }

    else if (!strcmp(cmd, "CH2_ON"))  { led_on('B', LED2); ch_state[1] = 1; }
    else if (!strcmp(cmd, "CH2_OFF")) { led_off('B', LED2); ch_state[1] = 0; }

    else if (!strcmp(cmd, "CH3_ON"))  { led_on('B', LED3); ch_state[2] = 1; }
    else if (!strcmp(cmd, "CH3_OFF")) { led_off('B', LED3); ch_state[2] = 0; }

    else if (!strcmp(cmd, "CH4_ON"))  { led_on('E', LED4); ch_state[3] = 1; }
    else if (!strcmp(cmd, "CH4_OFF")) { led_off('E', LED4); ch_state[3] = 0; }

    else if (!strcmp(cmd, "ALL_ON"))
    {
        leds_all_on();
        for (uint8_t i = 0; i < 4; i++) ch_state[i] = 1;
        lcd_show_message("ALL ON");
        _delay_ms(800);
    }
    else if (!strcmp(cmd, "ALL_OFF"))
    {
        leds_all_off();
        for (uint8_t i = 0; i < 4; i++) ch_state[i] = 0;
        lcd_show_message("ALL OFF");
        _delay_ms(800);
    }

    lcd_show_status();
}



void lcd_show_status(void)
{
    char line1[17];
    char line2[17];

    snprintf(line1, 17, "CH1:%s CH2:%s",
             ch_state[0] ? "ON " : "OFF",
             ch_state[1] ? "ON " : "OFF");

    snprintf(line2, 17, "CH3:%s CH4:%s",
             ch_state[2] ? "ON " : "OFF",
             ch_state[3] ? "ON " : "OFF");

    lcd_clear();
    lcd_set_cursor(0, 0);
    lcd_write_string(line1);
    lcd_set_cursor(1, 0);
    lcd_write_string(line2);
}


void handle_pirs(void)
{
    if (pir1_is_active()) {
        rooms[0].state = 1;
        rooms[0].last_motion_time = system_time_ms;
        led_on('B', LED1);
    }

    if (pir2_is_active()) {
        rooms[1].state = 1;
        rooms[1].last_motion_time = system_time_ms;
        led_on('B', LED2);
    }

    if (pir3_is_active()) {
        rooms[2].state = 1;
        rooms[2].last_motion_time = system_time_ms;
        led_on('B', LED3);
    }

    if (pir4_is_active()) {
        rooms[3].state = 1;
        rooms[3].last_motion_time = system_time_ms;
        led_on('E', LED4);
    }
}


void handle_timeouts(void)
{
    for (uint8_t i = 0; i < 4; i++) {
        if (rooms[i].state &&
            (system_time_ms - rooms[i].last_motion_time) > PIR_TIMEOUT_MS)
        {
            rooms[i].state = 0;

            if(ch_state[i] == 0) { // N'éteint la LED que si le canal n'est pas forcé ON
            switch (i) {
                case 0: led_off('B', LED1); break;
                case 1: led_off('B', LED2); break;
                case 2: led_off('B', LED3); break;
                case 3: led_off('E', LED4); break;
            }
        }
    }
    }
}
