#include "lcd_private.h"
#include "lcd.h"

static void lcd_write_nibble(uint8_t nibble) {
    LCD_EN_PORT &= ~(1 << LCD_EN_PIN);
    
    if (nibble & 0x08) LCD_D7_PORT |= (1 << LCD_D7_PIN);
    else LCD_D7_PORT &= ~(1 << LCD_D7_PIN);
    
    if (nibble & 0x04) LCD_D6_PORT |= (1 << LCD_D6_PIN);
    else LCD_D6_PORT &= ~(1 << LCD_D6_PIN);
    
    if (nibble & 0x02) LCD_D5_PORT |= (1 << LCD_D5_PIN);
    else LCD_D5_PORT &= ~(1 << LCD_D5_PIN);
    
    if (nibble & 0x01) LCD_D4_PORT |= (1 << LCD_D4_PIN);
    else LCD_D4_PORT &= ~(1 << LCD_D4_PIN);
    
    _delay_us(1);
    LCD_EN_PORT |= (1 << LCD_EN_PIN);
    _delay_us(1);
    LCD_EN_PORT &= ~(1 << LCD_EN_PIN);
    _delay_us(100);
}

static void lcd_write_byte(uint8_t data) {
    lcd_write_nibble(data >> 4);
    lcd_write_nibble(data & 0x0F);
}

static void lcd_power_up_4bit(void) {
    _delay_ms(40);
    lcd_write_nibble(0x03);
    _delay_ms(5);
    lcd_write_nibble(0x03);
    _delay_us(100);
    lcd_write_nibble(0x03);
    _delay_us(50);
    lcd_write_nibble(0x02);
    _delay_us(50);
}

void lcd_init(void) {
    // Configuration des broches en sortie
    LCD_D4_DDR |= (1 << LCD_D4_PIN);
    LCD_D5_DDR |= (1 << LCD_D5_PIN);
    LCD_D6_DDR |= (1 << LCD_D6_PIN);
    LCD_D7_DDR |= (1 << LCD_D7_PIN);
    LCD_RS_DDR |= (1 << LCD_RS_PIN);
    LCD_RW_DDR |= (1 << LCD_RW_PIN);
    LCD_EN_DDR |= (1 << LCD_EN_PIN);
    
    // Initialisation à LOW
    LCD_D4_PORT &= ~(1 << LCD_D4_PIN);
    LCD_D5_PORT &= ~(1 << LCD_D5_PIN);
    LCD_D6_PORT &= ~(1 << LCD_D6_PIN);
    LCD_D7_PORT &= ~(1 << LCD_D7_PIN);
    LCD_RS_PORT &= ~(1 << LCD_RS_PIN);
    LCD_RW_PORT &= ~(1 << LCD_RW_PIN);
    LCD_EN_PORT &= ~(1 << LCD_EN_PIN);
    
    // Séquence d'initialisation
    lcd_power_up_4bit();
    _delay_ms(50);
    
    // Configuration du LCD - ENVOYER PLUSIEURS FOIS
    lcd_write_command(0x28); // 4 bits, 2 lignes, 5x8 points
    _delay_ms(5);
    lcd_write_command(0x28);
    _delay_us(100);
    lcd_write_command(0x28);
    _delay_us(50);
    
    lcd_write_command(LCD_ON); // Display ON, cursor off
    lcd_write_command(LCD_INCR_RIGHT); // Increment, no shift
    lcd_clear();
    _delay_ms(50);
}

void lcd_write_command(uint8_t command) {
    LCD_RS_PORT &= ~(1 << LCD_RS_PIN);
    LCD_RW_PORT &= ~(1 << LCD_RW_PIN);
    lcd_write_byte(command);
    _delay_us(50);
}

void lcd_write_data(uint8_t data) {
    LCD_RS_PORT |= (1 << LCD_RS_PIN);
    LCD_RW_PORT &= ~(1 << LCD_RW_PIN);
    lcd_write_byte(data);
    _delay_us(50);
}

void lcd_clear(void) {
    lcd_write_command(LCD_CLEAR);
    _delay_ms(5); // Délai augmenté à 5ms
}

void lcd_set_cursor(uint8_t row, uint8_t col) {
    uint8_t address;
    uint8_t row_offsets[] = {0x00, 0x40};
    
    if (row >= NB_ROWS) row = NB_ROWS - 1;
    if (col >= NB_COLS) col = NB_COLS - 1;
    
    address = row_offsets[row] + col;
    lcd_write_command(LCD_ADDRSET | address);
}

void lcd_write_char(char c) {
    lcd_write_data(c);
}

void lcd_write_string(const char* str) {
    while (*str) {
        lcd_write_char(*str++);
    }
}

void lcd_animation(const char *message) {
    lcd_clear();
    lcd_set_cursor(0, 0);
    lcd_write_string("Mouvement ->");
    lcd_set_cursor(1, 0);

    //animation du texte caractere par caractere
    for(uint8_t i = 0; message[i] != '\0'; i++) {
        lcd_write_char(message[i]);
        _delay_ms(1000);
    }
}